﻿using System;
using System.Collections.Generic;
using InSimDotNet.Packets;
using System.Text;

namespace Dizplay_Cruise.Button
{
    public static class Sequence
    {
        public enum Sequences
        {
            DEFAULT_HUD = 1
        }

        /// <summary>
        /// Adds a button with these requirements to the Sequences.
        /// </summary>
        /// <param name="Button">The button for this part of the sequence.</param>
        /// <param name="Id">The Id of the whole sequence.</param>
        /// <param name="Name">The name of the whole sequence.</param>
        /// <param name="CustomId">The Custom Id of the sequence.</param>
        private static void AddToSequences(ButtonPacket Button, uint Id, string Name, string CustomId = "")
        {
            clsSequence S = new clsSequence();
            S.Button = Button;
            S.Id = Id;
            S.Name = Name;
            S.CustomId = CustomId;

            Dizplay.Sequences.Add(S);
        }

        /// <summary>
        /// Adds a button with these requirements to the Sequences.
        /// </summary>
        /// <param name="Id">The Id of the whole sequence.</param>
        /// <param name="Name">The name of the whole sequence.</param>
        /// <param name="CustomId">The Custom Id of the sequence.</param>
        /// <param name="Text">The text of the button.</param>
        /// <param name="ButtonStyles">Button style flags.</param>
        /// <param name="H">The height of the button (0 to 200).</param>
        /// <param name="W">The width of the button (0 to 200).</param>
        /// <param name="T">The distance from the top of the screen the button will be displayed (0 to 200).</param>
        /// <param name="L">The distance from the left of the screen the button will be displayed (0 to 200).</param>
        /// <param name="ClickID">Unique button click ID.</param>
        /// <param name="ReqI">Packet request ID.</param>
        /// <param name="UCID">The connection to display the button (0 = local / 255 = all).</param>
        /// <param name="xAlign">Horizontal button alignment (left, center, right)</param>
        /// <param name="yAlign">Vertical button alignment (top, middle, bottom)</param>
        /// <param name="TypeIn">The maximum characters the user is allowed to type in.</param>
        /// <param name="Caption">The caption for a type-in button.</param>
        /// <param name="Inst">Used internally by insim. (128 = always on top)</param>
        public static void AddToSequences(uint Id, string Name, string Creator, string Text, ButtonStyles ButtonStyles, byte H, byte W, int T, int L, byte ClickID, byte ReqI, int UCID, string xAlign = "left", string yAlign = "top", byte TypeIn = 0, string Caption = "", bool Inst = false, bool Public = true, string CustomId = "", string FunctionGroup = "none", string Function = "none")
        {
            switch (xAlign)
            {
                case "center":
                    L = (byte)((100 - W / 2) + L);
                    break;

                case "right":
                    L = (byte)((200 - W) - L);
                    break;

            }

            switch (yAlign)
            {
                case "middle":
                    T = (byte)((100 - H / 2) + T);
                    break;

                case "bottom":
                    T = (byte)((200 - H) - T);
                    break;
            }

            ButtonPacket Button = new ButtonPacket { Styles = ButtonStyles, ClickId = ClickID, Height = H, Width = W, Left = (byte)L, Top = (byte)T, RequestId = ReqI, Text = Text, TypeIn = TypeIn, Caption = Caption, UseDefaultText = Inst, ConnectionId = UCID };

            clsSequence S = new clsSequence();
            S.Button = Processor.CopyButton(Button);
            S.Text = Text;
            S.Id = Id;
            S.Name = Name;
            S.Creator = Creator;
            S.Public = Public;
            S.CustomId = CustomId;
            S.Function = Function;

            Dizplay.Sequences.Add(S);
        }

        /// <summary>
        /// Creates a static sequence of buttons
        /// </summary>
        /// <param name="CS">Current Sequence</param>
        /// <param name="UCID">Unique Connection ID</param>
        public static void Queue(string Name, int UCID = 255, clsUser U = null, object SpecialValue = null, string FunctionGroup = "", bool DynamicsOnly = false, string SpecificFunction = "")
        {
            if (!Globals.Locked)
            {
                for (int i = 0; i < Dizplay.Sequences.Count; i++)
                {
                    clsSequence FS = Dizplay.Sequences[i];

                    clsSequence S = new clsSequence();
                    S.FunctionGroup = FS.FunctionGroup;
                    S.Function = FS.Function;
                    S.Name = FS.Name;
                    S.Text = FS.Text;
                    S.Button = Processor.CopyButton(FS.Button);
                    S.Button.ConnectionId = UCID;

                    if (S.Name == Name.ToLower())
                    {
                        bool Validated = false;
                        if ((U != null || SpecialValue != null) && (S.FunctionGroup == FunctionGroup || FunctionGroup == "") && (S.Function == SpecificFunction || SpecificFunction == ""))
                        {
                            switch (S.Function)
                            {
                                case "display_cash":
                                    Validated = true;
                                    if (SpecialValue == null)
                                    {
                                        S.Button.Text = S.Text
                                                .Replace("^%D_CASH^%", String.Format("{0:#,0.00}", ((decimal)U.Cash / 100)).Replace(',', ' ').Replace('.', ','));
                                    }
                                    else
                                    {
                                        S.Button.Text = S.Text
                                                .Replace("^%D_CASH^%", String.Format("{0:#,0.00}", (Convert.ToDecimal(SpecialValue) / 100)).Replace(',', ' ').Replace('.', ','));
                                    }
                                    break;

                                case "display_distance":
                                    Validated = true;
                                    if (SpecialValue == null)
                                    {
                                        S.Button.Text = S.Text
                                                .Replace("^%D_DISTANCE^%", String.Format("{0:#,0.00}", ((decimal)U.Distance / 65536000)).Replace(',', ' ').Replace('.', ','));
                                    }
                                    else
                                    {
                                        S.Button.Text = S.Text
                                                .Replace("^%D_DISTANCE^%", String.Format("{0:#,0.00}", (Convert.ToDecimal(SpecialValue) / 65536000)).Replace(',', ' ').Replace('.', ','));
                                    }
                                    break;

                                case "display_speedlimit":
                                    Validated = true;
                                    if (SpecialValue == null)
                                    {
                                        S.Button.Text = S.Text
                                                .Replace("^%D_SPEEDLIMIT^%", "" + Convert.ToInt32(U.CompCar.Speed.Kph));
                                    }
                                    else
                                    {
                                        S.Button.Text = S.Text
                                                .Replace("^%D_SPEEDLIMIT^%", "" + Convert.ToInt32(SpecialValue));
                                    }
                                    break;
                            }
                        }

                        if ((!DynamicsOnly && SpecificFunction == "") || Validated)
                        {
                            Processor.QueueButton(S.Button);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Creates a static sequence of buttons
        /// </summary>
        /// <param name="CS">Current Sequence</param>
        /// <param name="UCID">Unique Connection ID</param>
        public static void Create(Sequences CS, byte UCID = 255)
        {
            switch (CS)
            {
                case Sequences.DEFAULT_HUD:
                    AddToSequences
                    (
                        0, "default_hud", "broken", "", ButtonStyles.Light,
                        20, 64, 1, 0, 0, 41, UCID, xAlign: "center", FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "", ButtonStyles.Dark,
                        5, 66, 0, 0, 1, 41, UCID, xAlign: "center", FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "UF1 XFG ^7XRG LX4 LX6 RB4 ^8FXO XRT ^7RAC ^8VWS FZ5 ^7UFR XFR FXR XRR FZR ^8MRT",
                        (ButtonStyles)1,
                        4, 66, 1, 67, 2, 41, UCID, Inst: true, FunctionGroup: "hud", Function: "display_cars"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "Dizplay",
                        ButtonStyles.Light |
                        (ButtonStyles)5,
                        6, 11, 6, 69, 3, 41, UCID, Inst: true, FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "",
                        ButtonStyles.Dark |
                        ButtonStyles.Left |
                        (ButtonStyles)5,
                        6, 50, 6, 81, 4, 41, UCID, FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^2€ ^8^%D_CASH^%",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 25, 7, 81, 5, 41, UCID, FunctionGroup: "hud", Function: "display_cash"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^%D_DISTANCE^%km",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 25, 7, 106, 6, 41, UCID, FunctionGroup: "hud", Function: "display_distance"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "",
                        ButtonStyles.Dark |
                        ButtonStyles.Left |
                        (ButtonStyles)5,
                        6, 62, 13, 69, 7, 41, UCID, FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^%D_LOCATION^%",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 37, 14, 69, 8, 41, UCID, FunctionGroup: "hud", Function: "display_location"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^%D_SPEEDLIMIT^%km/h",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 25, 14, 106, 9, 41, UCID, FunctionGroup: "hud", Function: "display_speedlimit"
                    );
                    break;
            }
        }

        /// <summary>
        /// Queues a dynamically created sequence of buttons, taken from clsSequence
        /// </summary>
        /// <param name="CS">Current Sequence</param>
        /// <param name="UCID">Unique Connection ID</param>
        public static void Create(string CS, byte UCID = 255)
        {
            switch (CS.ToLower())
            {
                case "default_hud":
                    AddToSequences
                    (
                        0, "default_hud", "broken", "", ButtonStyles.Light,
                        20, 64, 1, 0, 0, 41, UCID, xAlign: "center", FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "", ButtonStyles.Dark,
                        5, 66, 0, 0, 1, 41, UCID, xAlign: "center", FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "UF1 XFG ^7XRG LX4 LX6 RB4 ^8FXO XRT ^7RAC ^8VWS FZ5 ^7UFR XFR FXR XRR FZR ^8MRT",
                        (ButtonStyles)1,
                        4, 66, 1, 67, 2, 41, UCID, Inst: true, FunctionGroup: "hud", Function: "display_cars"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "Dizplay",
                        ButtonStyles.Light |
                        (ButtonStyles)5,
                        6, 11, 6, 69, 3, 41, UCID, Inst: true, FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "",
                        ButtonStyles.Dark |
                        ButtonStyles.Left |
                        (ButtonStyles)5,
                        6, 50, 6, 81, 4, 41, UCID, FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^2€ ^8^%D_CASH^%",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 25, 7, 81, 5, 41, UCID, FunctionGroup: "hud", Function: "display_cash"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^%D_DISTANCE^%km",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 25, 7, 106, 6, 41, UCID, FunctionGroup: "hud", Function: "display_distance"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "",
                        ButtonStyles.Dark |
                        ButtonStyles.Left |
                        (ButtonStyles)5,
                        6, 62, 13, 69, 7, 41, UCID, FunctionGroup: "hud"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^%D_LOCATION^%",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 37, 14, 69, 8, 41, UCID, FunctionGroup: "hud", Function: "display_location"
                    );
                    AddToSequences
                    (
                        0, "default_hud", "broken", "^%D_SPEEDLIMIT^%km/h",
                        ButtonStyles.Left |
                        (ButtonStyles)1,
                        4, 25, 14, 106, 9, 41, UCID, FunctionGroup: "hud", Function: "display_speedlimit"
                    );
                    break;
            }
        }
    }
}
